#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#define MAX_ROWS 100
#define MAX_COLS 100

/* Reads the maze from a text file and puts it in input 2D array.
 * Returns the height and width of the maze to rows and cols 
 * respectively. Assumptions: Walls are '#', empty (walkable) space
 * is '*'. 'S' is a single walkable cell which corresponds to the
 * starting point. Similarly, 'E' is a single walkable cell which 
 * corresponds to the exit. The first two rows of the text file 
 * provide the height (in characters) and width of the maze */
int read_maze(char input[][MAX_COLS], int *rows, int *cols) {
	int i, j;
	
	/* Read height and width and check they are within the 
	 * maximum allowable range */
	scanf("%d", rows);
	scanf("%d", cols);
	if (*rows < 0 || *cols < 0 || *rows > MAX_ROWS || *cols > MAX_COLS) {
		return(-1);
	}
	
	/* Now read the maze and put it in the input 2D array */
	for (i = 0; i < *rows; i++) {
		for (j = 0; j < *cols; j++) {
			scanf(" %c", &input[i][j]);
		}
	}
	return(0);
}


void print_maze(char input[][MAX_COLS], int rows, int cols) {
	int i, j;
	
	for (i = 0; i < rows; i++) {
		for (j = 0; j < cols; j++)
			printf("%c", input[i][j]);
		printf("\n");
	}
}


int rec_solve(char input[][MAX_COLS], int rows, int cols, int start_row, int start_col) {
	int res;

	/* The following three lines create the "animation" of the
	 * solution process. If you just want to see the final output,
	 * put them in comments */
	//system("clear"); /* Clears the terminal screen */
	//print_maze(input, rows, cols); /* Print the current status of
	//								* the maze */
	//usleep(20000);	/* Do nothing for 20 msec to allow time for 
	//				 * the user to see the printout */

	if (input[start_row][start_col] == 'E') /* Congrats, we 've found 
											 * the exit, therefore the 
											 * solution */
		return(1);
	if (input[start_row][start_col] != '*' && 	/* We are at a cell where */
		input[start_row][start_col] != 'S')		/* we should not step */
		return(-1);								/* Report failure */
	input[start_row][start_col] = 'P'; 	/* Mark the cell as visited */
										/* This protects us from walking *
										 * in circles and at the same *
										 * time marks the path we have *
										 * taken up to now. When finishihg *
										 * 'P' marked cells will be the *
										 * path to the solution */
	if (start_row != 0) { /* Try going up if you can */
		res = rec_solve(input, rows, cols, start_row-1, start_col);
		if (res == 1) /* If by going up we eventually found a solution */
			return(1);/* report that to our caller */
		/* Otherwise we 'll continue to try going down from this cell */
	}
	if (start_row != rows-1) { /* Try going down if you can */
		res = rec_solve(input, rows, cols, start_row+1, start_col);
		if (res == 1)
			return(1);
	}
	if (start_col != 0) { /* Try going left if you can */
		res = rec_solve(input, rows, cols, start_row, start_col-1);
		if (res == 1)
			return(1);
	}
	if (start_col != cols-1) { /* Try going right if you can */
		res = rec_solve(input, rows, cols, start_row, start_col+1);
		if (res == 1)
			return(1);
	}
	/* If by trying all four directions starting from the current cell *
	 * we eventually failed, we 'll have to backtrack. Mark this cell *
	 * as non-visited and report failure to the one that called us */
	input[start_row][start_col] = '*';
	return(-1);
}


/* Find the coordinates of the starting cell ('S') and return them
 * in start_row, start_col */
void find_start(char input[][MAX_COLS], int rows, int cols, 
				int *start_row, int *start_col) {
	int i, j;
	
	for (i = 0; i < cols; i++)
		for (j = 0; j < cols; j++) 
			if (input[i][j] == 'S') {
				*start_row = i;
				*start_col = j;
				return;
			}
}


void solve_maze(char input[][MAX_COLS], int rows, int cols) {
	int start_row, start_col;
	
	find_start(input, rows, cols, &start_row, &start_col);		
	rec_solve(input, rows, cols, start_row, start_col);
}


int main(int argc, char *argv[]) {
	int width, length;
	
	char maze[MAX_ROWS][MAX_COLS];

	if (read_maze(maze, &length, &width) != 0) {
		printf("Problem reading input\n");
		return(1);
	}
	
	solve_maze(maze, length, width);
	
	system("clear"); /* Clears the terminal screen */
	print_maze(maze, length, width);
	return(0);
}
