function [index2D] = find_nodes_with_y(y0 , x_lower, x_upper, domain)
%Finds the i,j indices of one or more nodes with:
%   y = y0 AND x >= x_lower AND x <= x_upper.
%Input
% y0 : the exact y-coordinate of the target nodes
% x_lower, x_upper : lower and upper bounds for the x-coordinate of the 
%   target nodes. Only the nodes with x_lower <= x <= x_upper will be 
%   returned.
% domain : struct containing the properties of the domain
%Output
% index2D : (n x 2) matrix, where n is the number of nodes that satisfy
%   the conditions. Each row is the pair [i, j] of the node.

% Get the necessary properties
Lx =  domain.Lx;
Ly = domain.Ly;
nx = domain.nx;
ny = domain.ny;
dx = domain.dx;
dy = domain.dy;
tol = min(dx/2, dy/2);

% Check the coordinates
if (y0 <= -tol) || (y0 >= Ly+tol)
    msg = ['nodes with y0=', num2str(y0), ' are outside the plate'];
    throw(MException('find_nodes_with_y:invalid_coordinates', msg))
end
if (x_lower <= -tol) || (x_upper >= Lx+tol) || (x_lower >= x_upper)
    msg = 'x_lower, x_upper must be inside the plate and x_lower < x_upper';
    throw(MException('find_nodes_with_y:invalid_coordinates', msg))
end

% Find the (i,j) pair
index2D = zeros(0,2);
for i=1:nx
    for j=1:ny
        xi = (i-1)*dx;
        yj = (j-1)*dy;
        if (yj > y0-tol) && (yj < y0+tol) && ...
                (xi >= x_lower) && (xi <= x_upper)
            index2D = [index2D; [i, j]];
        end
    end
end

end

