function [node_funcs] = define_all_node_equations(node_supports, domain)
%For each node, defines the function handle that, when called later, 
%will produce the finite difference equations for that node
%Input
% node_supports: A matrix with dimensions (nj x ni). Each entry of this 
%   matrix contains a string that describes the type of support of node j,i.
%   Valid values are 'free', 'simple', 'supported' (and 'g' for ghost nodes).
% domain : struct containing the properties of the domain.
%Output
% node_funcs : Matrix with dimensions (nj x ni). Each entry j,i 
%   contains a function handle that, when called, will return the finite 
%   difference equations for node i,j. Entries j,i outside the plate  
%       contain 'g' instead of a function handle.

% Get the properties needed in this function
ni = domain.ni;
nj = domain.nj;

% Mark nodes outside the plate as 'g' for ghost
node_funcs = cell(nj,ni);
for i=1:ni
    for j=1:nj
        if (i >= 3) && (i <= ni-2) && (j >= 3) && (j <= nj-2)
            % Real nodes: do nothing.
        else
            node_funcs{j,i} = 'g';
        end
    end
end

% Internal nodes 
for i=4:ni-3
    for j=4:nj-3
        support = node_supports{j,i};
        node_funcs{j,i} = define_equations_internal(i, j, support);
    end
end

% Boundaries x = 0, x = Lx
i_at_boundary = [3, ni-2];
for I=1:2
    i = i_at_boundary(I);
    for j=4:nj-3
        support = node_supports{j,i};
        node_funcs{j,i} = define_equations_boundary(i, j, support, 'x');
    end
end

% Boundaries y = 0, y = Ly
j_at_boundary = [3, nj-2];
for J=1:2
    j = j_at_boundary(J);
    for i=4:ni-3
        support = node_supports{j,i};
        node_funcs{j,i} = define_equations_boundary(i, j, support, 'y');
    end
end

% Corners
cornerIndex2D = [3,3; ni-2,3; 3,nj-2; ni-2,nj-2];
for k=1:4
    i = cornerIndex2D(k,1);
    j = cornerIndex2D(k,2);
    support = node_supports{j,i};
    node_funcs{j,i} = define_equations_corner(i, j, support);
end


end

