function [node_supports] = apply_supports(support_text, nodes_index2D, ...
    node_supports)
%Applies 'fixed', 'simple' or 'free' supports at selected nodes.
%Input
% support_text : What type of support to apply. Must be 'fixed', 'simple'
%   or 'free'.
% nodes_index2D : Matrix with dimensions (n x 2) that contains the i,j
%   indices of the nodes. Each row is one node. Column 1 are the 
%   x-coordinates. Column 2 are the y-coordinates. 
% node_supports: A matrix with dimensions (nj x ni). Each entry of this 
%   matrix contains a string that describes the type of support of node j,i.
%   Valid values are 'free', 'simple', 'supported' (and 'g' for ghost nodes).
%Output
% node_supports : Updated version of the input "node_supports" 

n_nodes = size(nodes_index2D, 1);
if strcmp(support_text, 'fixed')
    
    % 'fixed' support overwrites anything else already existing at these 
    % nodes.
    for k=1:n_nodes
        i = nodes_index2D(k, 1);
        j = nodes_index2D(k, 2);
        node_supports{j,i} = 'fixed';
    end
    
elseif strcmp(support_text, 'simple')
    
    % 'simple' support overwrites 'free', but not 'fixed'
    for k=1:n_nodes
        i = nodes_index2D(k, 1);
        j = nodes_index2D(k, 2);
        existing_support = node_supports{j,i};
        if ~strcmp(existing_support, 'fixed')
            node_supports{j,i} = 'simple';
        end
    end
    
elseif strcmp(support_text, 'free')
    
    % 'free' cannot overwrite 'simple' or 'fixed' support. 
    % Thus, there is nothing to do here.
    
else
    
    msg = ['Support text must be "simple", "fixed" or "free", but was', ...
        support_text];
    throw(MException('set_supports:invalid_support', msg))
    
end

end
