function [e, xloc, yloc] = global_to_local_coordinates(xg, yg, domain)
%Find the element 'e' that contains the point with global coordinates(xg,yg). 
%Also calculate the local coordinates (xloc, yloc), inside element 'e'.
% xg, yg : The global coordinates of the point.
% domain: A struct containing the properties of the plate and mesh.
%Output
% e : The element that contains (xg, yg). Note that e=1, 2, nelx*nely.
% xloc, yloc : The local coordinates of the point inside the element.

Lx = domain.Lx;
Ly = domain.Ly;
nelx = domain.nelx;
dx = domain.dx;
dy = domain.dy;
tol = min(dx/2, dy/2);

% Check that the coordinates are inside the plate
if (xg <= -tol) || (xg >= Lx+tol) || (yg <= -tol) || (yg >= Ly+tol)
    msg = ['node (', num2str(xg), ',', num2str(yg), ') is outside the plate'];
    throw(MException('find_node_with_xy:invalid_coordinates', msg))
end

% Bound the point inside the plate if it is outside (but within tolerance)
if (xg < 0)
    xg = 0;
elseif (xg > Lx)
    xg = Lx;
end
if (yg < 0)
    yg = 0;
elseif (yg > Ly)
    yg = Ly;
end

% Find the element
ex = max(ceil(xg/dx), 1);
ey = max(ceil(yg/dy), 1);
e = (ey-1) * nelx + ex;

% Find the local coordinates of the point within that element
a = dx / 2;
b = dy / 2;
xloc = xg - (ex-1)*dx - a;
yloc = yg - (ey-1)*dy - b;

end

