function [index2D] = find_nodes_with_x(x0 , y_lower, y_upper, domain)
%Finds the i,j indices of one or more nodes with:
%   x = x0 AND y >= y_lower AND y <= y_upper.
%Input
% x0 : the exact x-coordinate of the target nodes
% y_lower, y_upper : lower and upper bounds for the y-coordinate of the 
%   target nodes. Only the nodes with y_lower <= y <= y_upper will be 
%   returned.
% domain : struct containing the properties of the domain
%Output
% index2D : (n x 2) matrix, where n is the number of nodes that satisfy
%   the conditions. Each row is the pair [i, j] of the node.

% Get the necessary properties
Lx =  domain.Lx;
Ly = domain.Ly;
nx = domain.nx;
ny = domain.ny;
dx = domain.dx;
dy = domain.dy;
tol = min(dx/2, dy/2);

% Check the coordinates
if (x0 <= -tol) || (x0 >= Lx+tol)
    msg = ['nodes with x0=', num2str(x0), ' are outside the plate'];
    throw(MException('find_nodes_with_x:invalid_coordinates', msg))
end
if (y_lower <= -tol) || (y_upper >= Ly+tol) || (y_lower >= y_upper)
    msg = 'y_lower, y_upper must be inside the plate and y_lower < y_upper';
    throw(MException('find_nodes_with_x:invalid_coordinates', msg))
end

% Find the (i,j) pair
index2D = zeros(0,2);
for i=1:nx
    for j=1:ny
        xi = (i-1)*dx;
        yj = (j-1)*dy;
        if (xi > x0-tol) && (xi < x0+tol) && ...
                (yj >= y_lower) && (yj <= y_upper)
            index2D = [index2D; [i, j]];
        end
    end
end

end

