function [A_act, active_w, inactive_w] = keep_active_unknowns(A)
%Removes from the list of unknowns, those w of ghost nodes that are not 
%used in any finite difference equations. This are called inactive unknowns.
%In contrast, active unknowns are the unknowns of real nodes and ghost nodes
%that are actually used in at least 1 equation. This procedure is necessary
%to get a linear system with as many equations as unknowns, which can then 
%be solved.
%Input
% A: matrix containing the left-hand-side coefficients of the finite 
%   difference equations. Its dimensions are (n_equations x n_unknowns).
%   Some of these columns correspond to inactive unknowns
%Output
% A_act : The matrix A, without the columns corresponding to inactive
% ghost unknowns
% active_w : A vector containing the original unknowns that are active. 
%   Its dimensions are (1 x n_active_unknowns)
% inactive_w : A vector containing the original unknowns that are inactive. 
%   Its dimensions are (1 x n_inactive_unknowns)

ne = size(A,1); % number of equations e
ns = size(A,2); % number of unknowns s

% Find active and inactive unknowns w
active_w = [];
inactive_w = [];
for s=1:ns
    is_active = 0; % 1 = active unknown, 0 = inactive unknown
    for e=1:ne
        if A(e,s) ~= 0
            is_active = 1;
            break;
        end
    end
    if is_active == 1
        active_w = [active_w, s];
    else
        inactive_w = [inactive_w, s];
    end
end

% New matrix A_act with only the coefficients of the active unknowns
A_act = A(:, active_w);

end

