function [domain, node_supports] = initialize_domain(D, v, Lx, Ly, nx, ny)
%Helps the user to specify all properties of the domain (plate and grid).
%Input
% D : Flexural stiffness of the plate
% v : Poisson's ratio
% Lx : The length of the plate along axis x
% Ly : The length of the plate along axis y
% nx : The number of nodes (real ones, not ghost nodes) along axis x
% ny : The number of nodes (real ones, not ghost nodes) along axis y
%Output
% domain: A struct containing the properties of the domain. It can be 
%   passed to other functions, but must not be changed. In addition to the 
%   input it also contains:
%   h : The distance between 2 nodes along axis x. Is equal to the 
%       distance between 2 nodes along along axis y.
%   ni : The number of all nodes (real and ghost ones) along axis x
%   nj : The number of all nodes (real and ghost ones) along axis y.
% node_supports: A matrix with dimensions (nj x ni). Each entry of this 
%   matrix contains a string that describes the type of support of node j,i.
%   Valid values are 'free', 'simple', 'supported' and 'g'. After this 
%   function ends, all real nodes have the value 'free', which can then be
%   changed by other functions. Ghost nodes always have the value 'g'.

% Check that dx = dy = h
dx = Lx / (nx-1);
dy = Ly / (ny-1);
if (dx ~= dy)
    msg = 'Lx/(nx-1) must be equal to Ly/(ny-1)'; 
    throw(MException('define_domain:invalid_grid', msg));
end

% Store the properties in a struct 
ni = nx+4;
nj = ny+4;
domain.ni = ni;
domain.nj = nj;
domain.h = dx;
domain.D = D;
domain.v = v;
domain.Lx = Lx;
domain.Ly = Ly;
domain.nx = nx;
domain.ny = ny;

% Initially all nodes of the plate are free
node_supports = cell(nj, ni);
for i=1:ni
    for j=1:nj
        if (i >= 3) && (i <= ni-2) && (j >= 3) && (j <= nj-2)
            % Mark nodes inside the plate and on its boundary as 'free'. 
            % Supports can be added later.
            node_supports{j,i} = 'free';
        else
            % Mark nodes outside the plate as 'g' for ghost
            node_supports{j,i} = 'g';
        end
    end
end

end