function [coefficients, rhs] = equations_internal_free(i, j, domain, qij)
%Creates FD equations for the equilibrium of an internal node i,j
%Input
% i,j : The 2D indices of the node. Values i=1,2,ni-1,ni and j=1,2,nj-1,nj 
%   correspond to ghost nodes outside the plate.
% domain : struct containing the properties of the domain
% qij : the equivalent load at the node
%Output
% coefficients: matrix with as many rows as the equations for this node and
%   as many columns as the unknowns (which are as many as the nodes). 
%   This matrix contains the left-hand-side coefficients that are  
%   multiplied with the unknowns. 
% rhs: vector with as many rows as the equations for this node. This vector
%   contains the right-hand-side constants of the equations. 

% Get the properties needed in this function
ni = domain.ni;
nj = domain.nj;
h = domain.h;
D = domain.D;

% 1 equation per node
coefficients = zeros(1, ni*nj);
rhs = zeros(1,1);

% 1D index of each node on the stencil
s_c = (j-1)*ni+i;

s_n = (j-2)*ni+i;
s_s = j*ni+i;
s_w = (j-1)*ni+i-1;
s_e = (j-1)*ni+i+1;

s_nw = (j-2)*ni+i-1;
s_ne = (j-2)*ni+i+1;
s_sw = j*ni+i-1;
s_se = j*ni+i+1;

s_nn = (j-3)*ni+i;
s_ss = (j+1)*ni+i;
s_ww = (j-1)*ni+i-2;
s_ee = (j-1)*ni+i+2;

% Left-hand-side(s) (coefficients) of the equation(s)
coefficients(1,s_nn) = 1;
coefficients(1,s_nw) = 2;
coefficients(1,s_n) = -8;
coefficients(1,s_ne) = 2;
coefficients(1,s_ww) = 1;
coefficients(1,s_w) = -8;
coefficients(1,s_c) = 20;
coefficients(1,s_e) = -8;
coefficients(1,s_ee) = 1;
coefficients(1,s_sw) = 2;
coefficients(1,s_s) = -8;
coefficients(1,s_se) = 2;
coefficients(1,s_ss) = 1;

% Right-hand-side(s) (constants) of the equation(s)
rhs(1,1) = qij*h^4/D;

end