function [coefficients, rhs] = equations_boundary_free_y(i, j, domain, qij)
%Creates FD equations for a boundary node i,j on a free edge
%   perpendicular to axis y.
%Input
% i,j : The 2D indices of the node. Values i=1,2,ni-1,ni and j=1,2,nj-1,nj 
%   correspond to ghost nodes outside the plate.
% domain : struct containing the properties of the domain
% qij : the equivalent load at the node
%Output
% coefficients: matrix with as many rows as the equations for this node and
%   as many columns as the unknowns (which are as many as the nodes). 
%   This matrix contains the left-hand-side coefficients that are  
%   multiplied with the unknowns. 
% rhs: vector with as many rows as the equations for this node. This vector
%   contains the right-hand-side constants of the equations. 

% Get the properties needed in this function
ni = domain.ni;
nj = domain.nj;
h = domain.h;
D = domain.D;
v = domain.v;

% 3 equations per node
coefficients = zeros(3, ni*nj);
rhs = zeros(3,1);

% 1D index of each node on the stencil
s_c = (j-1)*ni+i;

s_n = (j-2)*ni+i;
s_s = j*ni+i;
s_w = (j-1)*ni+i-1;
s_e = (j-1)*ni+i+1;

s_nw = (j-2)*ni+i-1;
s_ne = (j-2)*ni+i+1;
s_sw = j*ni+i-1;
s_se = j*ni+i+1;

s_nn = (j-3)*ni+i;
s_ss = (j+1)*ni+i;
s_ww = (j-1)*ni+i-2;
s_ee = (j-1)*ni+i+2;

% 1st equation: V^4 w = q*h^4/D
coefficients(1,s_nn) = 1;
coefficients(1,s_nw) = 2;
coefficients(1,s_n) = -8;
coefficients(1,s_ne) = 2;
coefficients(1,s_ww) = 1;
coefficients(1,s_w) = -8;
coefficients(1,s_c) = 20;
coefficients(1,s_e) = -8;
coefficients(1,s_ee) = 1;
coefficients(1,s_sw) = 2;
coefficients(1,s_s) = -8;
coefficients(1,s_se) = 2;
coefficients(1,s_ss) = 1;
rhs(1,1) = qij*h^4/D;

% 2nd equation: My = 0
coefficients(2,s_n) = 1;
coefficients(2,s_w) = v;
coefficients(2,s_c) = -2-2*v;
coefficients(2,s_e) = v;
coefficients(2,s_s) = 1;
rhs(2,1) = 0;

% 3rd equation: Ry = 0
coefficients(3,s_nn) = -1;
coefficients(3,s_nw) = v-2;
coefficients(3,s_n) = 6-2*v;
coefficients(3,s_nw) = v-2;
coefficients(3,s_sw) = 2-v;
coefficients(3,s_s) = 2*v-6;
coefficients(3,s_se) = 2-v;
coefficients(3,s_ss) = +1;
rhs(3,1) = 0;

end

